(function() {

  var TodoItem = Backbone.Model.extend({});
  
  var TodoItemCollection = Backbone.Collection.extend({
    model: TodoItem,
    localStorage: new Backbone.LocalStorage("TodoStorage"),
    completed: function () {
      return this.filter(function (todo) {
        return todo.get('completed');
      });
    },
    remaining: function () {
      return this.filter(function (todo) {
        return !todo.get('completed');
      });
    },
    toggleAll: function(status) {
      this.each(function(item) {
        item.save({completed: status})
      });
    },
    clearCompleted: function() {
      this.chain().select(function(item) {
        return item.get('completed');
      }).each(function(model) {
        model.destroy();
      });
    },
    nextOrder: function () {
      if (!this.length) {
        return 1;
      }
      return this.last().get('order') + 1;
    },
    comparator: function (todo) {
      return todo.get('order');
    }
    
  });
 
  var TodoItemView = Backbone.View.extend({
    tagName: 'li',
    template: _.template($('#item-template').html()), 
    events: {
      'click .toggle': 'toggleCompleted',
      'dblclick label': 'edit',
      'click .destroy': 'clear',
      'keypress .edit': 'updateOnEnter',
      'blur .edit': 'close'
    },
    initialize: function () {
      this.listenTo(this.model, 'change', this.render);
      this.listenTo(this.model, 'destroy', this.remove);
      this.listenTo(this.model, 'visible', this.toggleVisible);
    },
    render: function () {
      this.$el.html(this.template(this.model.toJSON()));
      this.$el.toggleClass('completed', this.model.get('completed'));
      this.$input = this.$('.edit');
      return this;
    },
    toggleCompleted: function () {
      // TODO: Move to model
      this.model.save({completed: !this.model.get('completed')});
    },
    
    toggleVisible: function (filter) {
      this.$el.toggleClass('hidden', this.isHidden(filter));
    },
    isHidden: function (filter) {
      var isCompleted = this.model.get('completed');
      return (// hidden cases only
        (!isCompleted && filter === 'completed') ||
        (isCompleted && filter === 'active')
      );
    },
    
    edit: function () {
      this.$el.addClass('editing');
      this.$input.focus();
    },
    close: function () {
      var value = this.$input.val();
      var trimmedValue = value.trim();

      if (trimmedValue) {
        this.model.save({ title: trimmedValue });

        if (value !== trimmedValue) {
          // Model values changes consisting of whitespaces only are not causing change to be triggered
          // Therefore we've to compare untrimmed version with a trimmed one to chech whether anything changed
          // And if yes, we've to trigger change event ourselves
          this.model.trigger('change');
        }
      } else {
        this.clear();
      }

      this.$el.removeClass('editing');
    },
    // If you hit `enter`, we're through editing the item.
    updateOnEnter: function (e) {
      if (e.which === 13) {
        this.close();
      }
    },

    // Remove the item, destroy the model from *localStorage* and delete its view.
    clear: function () {
      this.model.destroy();
    }
  });
 
  var AppView = Backbone.View.extend({
    el: '#todoapp',
    statsTemplate: _.template($('#footer-template').html()),
    events: {
      'keypress #new-todo': 'createOnEnter',
      'click #clear-completed': 'clearCompleted',
      'click #toggle-all': 'toggleAllComplete'
    },
   
    initialize: function () {
      this.allCheckbox = this.$('#toggle-all')[0];
      this.$input = this.$('#new-todo');
      this.$footer = this.$('#footer');
      this.$main = this.$('#main');

      this.listenTo(this.collection, 'add', this.addOne);
      this.listenTo(this.collection, 'reset', this.addAll);
      this.listenTo(this.collection, 'change:completed', this.filterOne);
      this.listenTo(this.collection, 'filter', this.filterAll);
      this.listenTo(this.collection, 'all', this.render);
      
      // Suppresses 'add' events with {reset: true} and prevents the app view 
      // from being re-rendered for every model. Only renders when the 'reset'
      // event is triggered at the end of the fetch.
      this.collection.fetch({reset: true});
      
      this.filter = 'all';
      
    },
 
    render: function () {
      var completed = this.collection.completed().length;
      var remaining = this.collection.remaining().length;

      if (this.collection.length) {
        this.$main.show();
        this.$footer.show();

        this.$footer.html(this.statsTemplate({
          completed: completed,
          remaining: remaining,
          itemsPluralForm: (remaining == 1) ? "item" : "items",
          showFilters: this.collection.size() > 0 && (completed > 0),
          filters: [
            {name: 'All', value: 'all', className: this.filter === 'all' ? 'selected' : '' },
            {name: 'Active', value: 'active', className: this.filter === 'active' ? 'selected' : '' },
            {name: 'Completed', value: 'completed', className: this.filter === 'completed' ? 'selected' : '' }
          ]          
        }));

      } else {
        this.$main.hide();
        this.$footer.hide();
      }

      this.allCheckbox.checked = !remaining;
    },
    addOne: function (todo) {
      var view = new TodoItemView({ model: todo });
      $('#todo-list').append(view.render().el);
    },

    // Add all items in the **Todos** collection at once.
    addAll: function () {
      this.$('#todo-list').html('');
      this.collection.each(this.addOne, this);
    },

    filterOne: function (todo) {
      todo.trigger('visible', this.filter);
    },

    filterAll: function (filter) {
      this.filter = filter;
      this.collection.each(this.filterOne, this);
      this.render();
    },
    // Generate the attributes for a new Todo item.
    newAttributes: function () {
      return {
        title: this.$input.val().trim(),
        order: this.collection.nextOrder(),
        completed: false
      };
    },

    // If you hit return in the main input field, create new **Todo** model,
    // persisting it to *localStorage*.
    createOnEnter: function (e) {
      if (e.which !== 13 || !this.$input.val().trim()) {
        return;
      }

      this.collection.create(this.newAttributes());
      this.$input.val('');
    },

    // Clear all completed todo items, destroying their models.
    clearCompleted: function () {
      this.collection.clearCompleted();
      return false;
    },

    toggleAllComplete: function () {
      var completed = this.allCheckbox.checked;

      this.collection.each(function (todo) {
        todo.save({
          'completed': completed
        });
      });
    }    
  });
  
  var TodoRouter = Backbone.Router.extend({
    routes: {
      '*filter': 'setFilter'
    },
    initialize: function(options) {
      this.view = options.view;
      this.todos = options.todos;
    },
    setFilter: function (param) {
      this.todos.trigger('filter', param);
    }
  });
    
  window.addEventListener('load', windowLoadHandler, false);
  function windowLoadHandler() {
    var todos = new TodoItemCollection();
    var view = new AppView({collection: todos});
    var router = new TodoRouter({view: view, todos: todos});
    Backbone.history.start({pushState: true});    
  }
}());
